<?php
function sha256_hex(string $data): string { return hash('sha256', $data); }
function hmac_sha256_raw(string $key, string $msg): string {
    return hash_hmac('sha256', $msg, $key, true);
}
function shuffleProvably(array $deck, string $serverSeed, string $clientSeed, int $nonce): array {
    for ($i = count($deck) - 1; $i > 0; $i--) {
        $msg = $clientSeed . ':' . $nonce . ':' . $i;
        $h = hmac_sha256_raw($serverSeed, $msg);
        $r = unpack('N', substr($h, 0, 4))[1];
        $j = $r % ($i + 1);
        $tmp = $deck[$i]; $deck[$i] = $deck[$j]; $deck[$j] = $tmp;
    }
    return $deck;
}
function betAt(int $k, int $bet0, float $alpha=0.20, float $cap=4.0): int {
    $linear = (int) round($bet0 * (1 + $alpha * $k));
    $capVal = (int) round($bet0 * $cap);
    return min($linear, $capVal);
}
function applyDraw(array $hand5, array $holds, array $remaining47, array &$takenMask47, array $picks): array {
    $new = $hand5; $idx = 0;
    for ($pos=0; $pos<5; $pos++) {
        if (!empty($holds[$pos])) continue;
        $target = $picks[$idx++] ?? null;
        if ($target===null || $target<0 || $target>=47 || !empty($takenMask47[$target])) {
            http_response_code(400);
            die(json_encode(['error' => 'Pick inválido']));
        }
        $new[$pos] = $remaining47[$target];
        $takenMask47[$target] = true;
    }
    return $new;
}
function card_rank(int $id): int { return $id % 13; }       // 0=A,1=2,...,12=K
function card_suit(int $id): int { return intdiv($id, 13); } // 0..3
function counts_by_rank(array $hand5): array {
    $cnt = array_fill(0, 13, 0);
    foreach ($hand5 as $c) { $cnt[card_rank($c)]++; }
    return $cnt;
}
function counts_by_suit(array $hand5): array {
    $cnt = array_fill(0, 4, 0);
    foreach ($hand5 as $c) { $cnt[card_suit($c)]++; }
    return $cnt;
}
function is_flush(array $hand5): bool {
    foreach (counts_by_suit($hand5) as $v) if ($v==5) return true;
    return false;
}
function is_straight(array $hand5): bool {
    $r = [];
    foreach ($hand5 as $c) $r[] = card_rank($c);
    sort($r);
    $r = array_values(array_unique($r));
    if (count($r) != 5) return false;
    if ($r[4] - $r[0] == 4) return true;
    return $r === [0,1,2,3,4]; // A-2-3-4-5
}
function hand_rank_and_mult(array $hand5): array {
    $flush = is_flush($hand5);
    $straight = is_straight($hand5);
    $ranks = counts_by_rank($hand5);
    rsort($ranks);

    if ($flush && $straight) {
        $rs = [];
        foreach ($hand5 as $c) $rs[] = card_rank($c);
        sort($rs);
        $set = array_unique($rs); sort($set);
        if ($set === [0,9,10,11,12]) return ['rank'=>'ROYAL_FLUSH','mult'=>250];
        return ['rank'=>'STRAIGHT_FLUSH','mult'=>50];
    }
    if ($ranks[0]==4) return ['rank'=>'FOUR','mult'=>25];
    if ($ranks[0]==3 && $ranks[1]==2) return ['rank'=>'FULL_HOUSE','mult'=>9];
    if ($flush) return ['rank'=>'FLUSH','mult'=>6];
    if ($straight) return ['rank'=>'STRAIGHT','mult'=>4];
    if ($ranks[0]==3) return ['rank'=>'THREE','mult'=>3];
    if ($ranks[0]==2 && $ranks[1]==2) return ['rank'=>'TWO_PAIR','mult'=>2];

    $cnt = counts_by_rank($hand5);
    foreach ([0,10,11,12] as $hi) if ($cnt[$hi] == 2) return ['rank'=>'JACKS_OR_BETTER','mult'=>1];
    return ['rank'=>'NOTHING','mult'=>0];
}
function paytable(): array {
    return [
      'ROYAL_FLUSH'=>250,
      'STRAIGHT_FLUSH'=>50,
      'FOUR'=>25,
      'FULL_HOUSE'=>9,
      'FLUSH'=>6,
      'STRAIGHT'=>4,
      'THREE'=>3,
      'TWO_PAIR'=>2,
      'JACKS_OR_BETTER'=>1,
      'NOTHING'=>0
    ];
}

/** ---------- PROSPECTS baseados APENAS nas cartas em HOLD (heurística) ---------- */
function held_cards(array $hand5, array $holds): array {
    $out=[]; for($i=0;$i<5;$i++){ if(!empty($holds[$i])) $out[]=$hand5[$i]; } return $out;
}
function is_four_to_straight(array $ranksHeld): bool {
    $ranksHeld = array_values(array_unique($ranksHeld)); sort($ranksHeld);
    // verifica se existe subsequência de 4 ranks consecutivos
    if (count($ranksHeld) < 4) return false;
    for ($i=0; $i+3<count($ranksHeld); $i++){
        if ($ranksHeld[$i+3]-$ranksHeld[$i]==3) return true;
    }
    // caso wheel A-2-3-4
    $set = $ranksHeld;
    return $set === [0,1,2,3] || $set === [1,2,3,4] || $set === [0,10,11,12]; // aproximações
}
function is_royal_component(int $rank): bool {
    return in_array($rank, [0,9,10,11,12], true);
}
function possible_outcomes(array $hand5, array $holds): array {
    $pt = paytable();
    $H = held_cards($hand5,$holds);
    if (count($H)==0) return []; // nada fixado → não sugere
    // contagens no HOLD
    $cntR = array_fill(0,13,0); $cntS = array_fill(0,4,0);
    foreach($H as $c){ $cntR[card_rank($c)]++; $cntS[card_suit($c)]++; }
    rsort($cntR); $maxSuit = max($cntS);

    $suggest = [];

    // Pares / Trincas / Dois Pares no HOLD
    if ($cntR[0]==2 && $cntR[1]<2){ // exatamente um par
        $suggest[]='THREE'; $suggest[]='FULL_HOUSE'; $suggest[]='FOUR';
    } elseif ($cntR[0]==3){ // trinca
        $suggest[]='FULL_HOUSE'; $suggest[]='FOUR';
    } elseif ($cntR[0]==2 && $cntR[1]==2){ // dois pares
        $suggest[]='FULL_HOUSE'; $suggest[]='FOUR';
    } elseif ($cntR[0]==1){ // nenhuma combinação, mas pode mirar Jacks+
        $suggest[]='JACKS_OR_BETTER';
    }

    // Draws de flush / straight
    if ($maxSuit>=4) $suggest[]='FLUSH';
    $ranksHeld=[]; foreach($H as $c) $ranksHeld[]=card_rank($c);
    if (is_four_to_straight($ranksHeld)) $suggest[]='STRAIGHT';

    // Componentes para straight flush / royal
    $royalSuitHit=false;
    foreach(range(0,3) as $s){
        $royalCount=0; foreach($H as $c){ if(card_suit($c)==$s && is_royal_component(card_rank($c))) $royalCount++; }
        if($royalCount>=4){ $royalSuitHit=true; break; }
    }
    if ($royalSuitHit){ $suggest[]='STRAIGHT_FLUSH'; $suggest[]='ROYAL_FLUSH'; }

    // Ordena por “valor”
    $order = array_keys($pt); // já é do maior para menor? (níveis altos primeiro)
    $suggest = array_values(array_unique($suggest));
    usort($suggest, function($a,$b) use($pt){ return $pt[$b] <=> $pt[$a]; });

    // remove NOTHING
    return array_values(array_filter($suggest, fn($x)=>$x!=='NOTHING'));
}
function prospects_with_payouts(array $hand5, array $holds, int $betBase): array {
    $pt = paytable();
    $list = possible_outcomes($hand5,$holds);
    $out = [];
    foreach($list as $rank){
        $mult = $pt[$rank] ?? 0;
        $out[] = ['rank'=>$rank,'mult'=>$mult,'payout'=>$mult*$betBase];
    }
    return $out;
}

/** ---------- Risco (já estava) ---------- */
function risk_score(array $hand5): int {
    $score = 0;
    $ev = hand_rank_and_mult($hand5);
    if ($ev['mult'] > 0) $score += min(10, $ev['mult']);
    foreach (counts_by_suit($hand5) as $v) if ($v == 4) { $score += 4; break; }
    $r = []; foreach ($hand5 as $c) $r[] = card_rank($c);
    sort($r); $r = array_values(array_unique($r));
    $maxRun=1; $run=1; for($i=1;$i<count($r);$i++){ if($r[$i]==$r[$i-1]+1){$run++; $maxRun=max($maxRun,$run);} else {$run=1;} }
    if ($maxRun >= 4) $score += 3;
    $cnt = counts_by_rank($hand5);
    foreach ([0,10,11,12] as $hi) if ($cnt[$hi]==2) { $score += 2; break; }
    for ($i=1;$i<=9;$i++) if ($cnt[$i]==2) { $score += 1; break; }
    return $score;
}
function risk_fee_for(array $hand5, int $betBase, float $baseRate, float $cap): int {
    $score = risk_score($hand5);
    $rate = min($cap, $baseRate * $score);
    return (int) floor($betBase * $rate);
}
