<?php
class CpfQuery {
  /* Utils */
  private static function d($s){ return preg_replace('/\D+/', '', (string)$s); }
  private static function cpf($s){ $d=self::d($s); return strlen($d)===11?$d:''; }
  private static function maskSocio($cpf11){ return '***'.substr($cpf11,3,6).'**'; }
  private static function norm($s){
    $s=strtoupper(trim((string)$s));
    if(function_exists('iconv')) $s=@iconv('UTF-8','ASCII//TRANSLIT//IGNORE',$s);
    return preg_replace('/[^A-Z0-9 ]+/','',$s);
  }
  private static function tok3($s){ return array_values(array_filter(preg_split('/\s+/',self::norm($s)),fn($x)=>strlen($x)>=3)); }

  private static function fetchAllNum(PDO $pdo, string $sql, array $binds=[]): array {
    $st=$pdo->prepare($sql);
    foreach($binds as $k=>$v){
      $st->bindValue($k,$v, is_int($v)?PDO::PARAM_INT:PDO::PARAM_STR);
    }
    $st->execute();
    return $st->fetchAll(PDO::FETCH_NUM) ?: [];
  }
  private static function fetchOneAssoc(PDO $pdo, string $sql, array $binds=[]): ?array {
    $st=$pdo->prepare($sql);
    foreach($binds as $k=>$v) $st->bindValue($k,$v, is_int($v)?PDO::PARAM_INT:PDO::PARAM_STR);
    $st->execute();
    $r=$st->fetch(PDO::FETCH_ASSOC);
    return $r?:null;
  }
  private static function joinPipeRows(array $rows): string {
    if(!$rows) return " (0 rows)";
    $out=[];
    foreach($rows as $r){
      $vals=array_map(fn($v)=> ($v===null)?'':(string)$v, $r);
      $out[]=" ".implode(" | ", $vals);
    }
    return implode("\n",$out);
  }
  private static function makeIn(string $prefix, array $vals, array &$binds): string {
    $ph=[]; $i=0;
    foreach($vals as $v){ $k=":{$prefix}{$i}"; $ph[]=$k; $binds[$k]=(string)$v; $i++; }
    return $ph?implode(',',$ph):"NULL";
  }

  public static function run(PDO $pdo, $args, $page=1, $per_page=500){
    $cpf=self::cpf($args);
    if(!$cpf){
      return ['text'=>"Informe um CPF válido (11 dígitos).",'info'=>"Parâmetro inválido",'page'=>1,'has_prev'=>false,'has_next'=>false];
    }
    $off=($page-1)*$per_page;
    $pdo->exec("SET statement_timeout TO '8s'");

    $sections=[];

    /* dados> — saída vertical "campo: valor" apenas para campos com conteúdo */
    $sections[] = "dados>";
    $nomeTitular = null; // ainda usado depois em socios>
    try {
      $assoc = self::fetchOneAssoc($pdo, "SELECT * FROM public.dados WHERE cpf = :cpf LIMIT 1", [":cpf"=>$cpf]);

      if ($assoc) {
        foreach ($assoc as $campo => $valor) {
          // só imprime se houver conteúdo
          if ($valor !== null) {
            $val = trim((string)$valor);
            if ($val !== '') {
              $sections[] = " " . $campo . ": " . $val;
            }
          }
          // captura primeiro campo de nome para filtro de sócios
          if ($nomeTitular === null && stripos($campo, 'nome') !== false && trim((string)$valor) !== '') {
            $nomeTitular = (string)$valor;
          }
        }
        // fallback de nome, se não achou acima
        if ($nomeTitular === null) {
          $nomeTitular = (string)($assoc['nome'] ?? $assoc['nome_pessoa'] ?? '');
        }
      } else {
        $sections[] = " (0 rows)";
      }
    } catch (\Throwable $e) {
      $sections[] = " (0 rows)";
      $nomeTitular = null;
    }
    $sections[] = "";

    /* telefone> (todas colunas) */
    $sections[]="telefone>";
    try{
      $rows=self::fetchAllNum($pdo,
        "SELECT telefone FROM public.telefones WHERE cpf = :cpf ORDER BY 1 LIMIT :lim OFFSET :off",
        [":cpf"=>$cpf,":lim"=>$per_page,":off"=>$off]
      );
      $sections[] = self::joinPipeRows($rows);
      // capturar lista de telefones para operadoras
      $tels=[];
      foreach($rows as $r){ foreach($r as $val){ /* heurística: pega o primeiro campo que parece telefone (>=8 dígitos) */
          $d=self::d($val);
          if(strlen($d)>=8){ $tels[]=$val; break; }
        } }
      $tels=array_values(array_unique($tels));
    }catch(\Throwable $e){ $sections[]=" (0 rows)"; $tels=[]; }
    $sections[]="";

    /* operadoras> — agrupar por idoper, pegar nome em public.operador e listar linhas completas por operadora */
    $sections[] = "operadoras>";
    try {
      // 1) Buscar todos os registros de operadoras_pf_raw deste CPF (ordenados por idoper)
      $st = $pdo->prepare("
        SELECT * 
          FROM public.operadoras_pf_raw
         WHERE cpf = :cpf
         ORDER BY idoper, ddd, telefone
         LIMIT :lim OFFSET :off
      ");
      $st->bindValue(':cpf', $cpf);
      $st->bindValue(':lim', $per_page, PDO::PARAM_INT);
      $st->bindValue(':off', $off, PDO::PARAM_INT);
      $st->execute();
      $rowsAssoc = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

      if (!$rowsAssoc) {
        $sections[] = " (0 rows)";
        $sections[] = "";
      } else {
        // 2) Agrupar por idoper
        $byIdoper = [];
        $idopers = [];
        foreach ($rowsAssoc as $r) {
          $idop = isset($r['idoper']) ? (string)$r['idoper'] : '';
          if ($idop === '') $idop = '0';
          $byIdoper[$idop][] = $r;
          $idopers[$idop] = true;
        }
        $idopers = array_keys($idopers);
        sort($idopers, SORT_NATURAL);

        // 3) Mapear idoper -> nome (tabela public.operador). Se não existir 'nome', usar o próprio idoper.
        $names = [];
        if ($idopers) {
          $binds = [];
          $ph = [];
          foreach ($idopers as $i => $v) { $ph[] = ":op$i"; $binds[":op$i"] = $v; }
          try {
            $q = $pdo->prepare("SELECT idoper, nome FROM public.operador WHERE idoper IN (".implode(',', $ph).")");
            foreach ($binds as $k => $v) $q->bindValue($k, $v);
            $q->execute();
            foreach ($q->fetchAll(PDO::FETCH_ASSOC) ?: [] as $r) {
              $names[(string)$r['idoper']] = (string)$r['nome'];
            }
          } catch (\Throwable $e) {
            // se a coluna for diferente (ex.: 'operadora'), vamos tentar esse nome também:
            try {
              $q = $pdo->prepare("SELECT idoper, operadora AS nome FROM public.operador WHERE idoper IN (".implode(',', $ph).")");
              foreach ($binds as $k => $v) $q->bindValue($k, $v);
              $q->execute();
              foreach ($q->fetchAll(PDO::FETCH_ASSOC) ?: [] as $r) {
                $names[(string)$r['idoper']] = (string)$r['nome'];
              }
            } catch (\Throwable $e2) {
              // fica sem nome (usa idoper mesmo)
            }
          }
        }

        // 4) Imprimir por operadora: "  {nome}>" e, abaixo, cada linha completa em pipe (na ordem do SELECT *)
        foreach ($idopers as $idop) {
          $label = isset($names[$idop]) && $names[$idop] !== '' ? strtolower($names[$idop]) : $idop;
          $sections[] = "  " . $label . ">";
          foreach ($byIdoper[$idop] as $r) {
            // preservar ordem do SELECT: usar array_values do FETCH_ASSOC
            $vals = array_map(function($v){ return $v === null ? '' : (string)$v; }, array_values($r));
            $sections[] = "\t" . implode(" | ", $vals);
          }
        }
        $sections[] = "";
      }
    } catch (\Throwable $e) {
      $sections[] = " (0 rows)";
      $sections[] = "";
    }
      
    /* cadsus> — pegar por CPF e imprimir todas as colunas */
    $sections[]="cadsus>";
    try{
      // 1) tentativa direta (coluna 'cpf' com 11 dígitos)
      $rows = self::fetchAllNum(
        $pdo,
        "SELECT * FROM public.cadsus WHERE cpf = :cpf LIMIT :lim OFFSET :off",
        [":cpf"=>$cpf, ":lim"=>$per_page, ":off"=>$off]
      );

      // 2) fallback: se vier vazio, tenta normalizar (caso a coluna seja text com pontuação)
      if (!$rows) {
        $rows = self::fetchAllNum(
          $pdo,
          "SELECT * FROM public.cadsus
            WHERE regexp_replace(cpf::text, '\\D', '', 'g') = :cpf
            LIMIT :lim OFFSET :off",
          [":cpf"=>$cpf, ":lim"=>$per_page, ":off"=>$off]
        );
      }

      $sections[] = self::joinPipeRows($rows);
    } catch(\Throwable $e) {
      $sections[] = " (0 rows)";
    }
    $sections[] = "";

    /* socios> — filtra por nome + máscara; imprime linha completa de socios */
    $sections[]="socios>";
    $cnpjs=[];
    try{
      $mask=self::maskSocio($cpf);
      $tokens=self::tok3($nomeTitular ?? '');
      if($tokens){
        // tentar com unaccent; se falhar, sem unaccent
        $conds=[]; $binds=[":mask"=>$mask];
        foreach($tokens as $i=>$tk){ $conds[]="unaccent(upper(nome_socio)) ILIKE :t$i"; $binds[":t$i"]="%$tk%"; }
        $sql="
          SELECT * FROM public.socios
           WHERE cnpj_cpf_socio = :mask
             AND ".implode(' AND ',$conds)."
        ";
        try{
          $rows=self::fetchAllNum($pdo,$sql,$binds);
          $rowsAssoc=self::fetchAllNum($pdo,$sql,$binds); // só pra mesma ordem; abaixo pegamos cnpj_basico de outra forma
        }catch(\Throwable $e){
          $conds=[]; $binds=[":mask"=>$mask];
          foreach($tokens as $i=>$tk){ $conds[]="upper(nome_socio) ILIKE :t$i"; $binds[":t$i"]="%$tk%"; }
          $sql="SELECT * FROM public.socios WHERE cnpj_cpf_socio = :mask AND ".implode(' AND ',$conds);
          $rows=self::fetchAllNum($pdo,$sql,$binds);
        }
        $sections[] = self::joinPipeRows($rows);

        // Para coletar cnpj_base das mesmas linhas (precisamos como texto, então refaço um SELECT focado)
        $binds=[":mask"=>$mask];
        $whereName=[];
        foreach($tokens as $i=>$tk){ $whereName[]="upper(nome_socio) ILIKE :t$i"; $binds[":t$i"]="%$tk%"; }
        $rs=$pdo->prepare("SELECT cnpj_basico FROM public.socios WHERE cnpj_cpf_socio=:mask AND ".implode(' AND ',$whereName));
        foreach($binds as $k=>$v){ $rs->bindValue($k,$v); }
        $rs->execute();
        foreach($rs->fetchAll(PDO::FETCH_NUM) as $r){ if(isset($r[0]) && $r[0]!==''){ $cnpjs[$r[0]]=true; } }
      } else {
        $sections[]=" (0 rows)";
      }
    }catch(\Throwable $e){ $sections[]=" (0 rows)"; }
    $sections[]="";

    /* estabelecimentos> — dos CNPJs filtrados; imprime todas colunas */
    $sections[]="estabelecimentos>";
    try{
      $keys=array_keys($cnpjs);
      if($keys){
        $binds=[]; $in=self::makeIn('cb',$keys,$binds);
        $rows=self::fetchAllNum($pdo,"SELECT * FROM public.estabelecimentos WHERE cnpj_base IN ($in)",$binds);
        $sections[] = self::joinPipeRows($rows);
      } else {
        $sections[]=" (0 rows)";
      }
    }catch(\Throwable $e){ $sections[]=" (0 rows)"; }
    $sections[]="";

    /* empresas> — dos CNPJs filtrados; imprime todas colunas */
    $sections[]="empresas>";
    try{
      $keys=array_keys($cnpjs);
      if($keys){
        $binds=[]; $in=self::makeIn('eb',$keys,$binds);
        $rows=self::fetchAllNum($pdo,"SELECT * FROM public.empresas WHERE cnpj_base IN ($in)",$binds);
        $sections[] = self::joinPipeRows($rows);
      } else {
        $sections[]=" (0 rows)";
      }
    }catch(\Throwable $e){ $sections[]=" (0 rows)"; }

    return [
      'text'=>implode("\n",$sections),
      'info'=>"Consulta CPF",
      'page'=>$page,'has_prev'=>$page>1,'has_next'=>false
    ];
  }
}