<?php
// /var/www/html/card/invite.php
ob_start();
session_start();

require __DIR__.'/db.php';
require __DIR__.'/token_helpers.php';   // generate_uuid(), create_user_token()
require __DIR__.'/locale_helpers.php';  // normalize_locale()
require __DIR__.'/i18n.php';

$cfg = include __DIR__.'/config.php';
$pdo = db();

// ---- helpers ----
function h($s){ return htmlspecialchars($s ?? '', ENT_QUOTES,'UTF-8'); }
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
function csrf(){ return $_SESSION['csrf']; }
function csrf_ok($t){ return !empty($_SESSION['csrf']) && hash_equals($_SESSION['csrf'], $t ?? ''); }

// ---- carrega convite ----
$code = $_GET['c'] ?? '';
if (!$code) { http_response_code(400); echo "Convite inválido."; exit; }

$st = $pdo->prepare("SELECT * FROM invites WHERE code=? LIMIT 1");
$st->execute([$code]);
$invite = $st->fetch();

if (!$invite) { http_response_code(404); echo "Convite não encontrado."; exit; }
if (($invite['status'] ?? 'pending') !== 'pending') { http_response_code(410); echo "Este convite não está mais ativo."; exit; }
if (!empty($invite['expires_at']) && strtotime($invite['expires_at']) < time()) {
  http_response_code(410); echo "Este convite expirou."; exit;
}

// idiomas suportados / default
$all_locales = $cfg['locales'] ?? ['pt-BR','en-US','es-419'];
$default_locale = $invite['default_locale'] ?: ($cfg['default_locale'] ?? 'pt-BR');

// prefill GET (opcional)
$prefill_name  = trim($_GET['name']  ?? '');
$prefill_email = trim($_GET['email'] ?? ($invite['email'] ?? ''));
$prefill_phone = trim($_GET['phone'] ?? '');
if (!empty($_GET['lang'])) { $default_locale = normalize_locale($_GET['lang']); }

$err = null;

// ---- POST: criar ou reaproveitar usuário e perfis ----
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['__create'])) {
  if (!csrf_ok($_POST['__csrf'] ?? '')) { http_response_code(403); echo "CSRF inválido."; exit; }

  $name    = trim($_POST['name'] ?? '');
  $email   = trim($_POST['email'] ?? ($invite['email'] ?? '')); // pode ser vazio
  $phone   = trim($_POST['phone'] ?? '');
  $locale0 = normalize_locale($_POST['default_locale'] ?? $default_locale);
  $langs   = $_POST['langs'] ?? []; // marcados

  if ($name === '') { $err = "Informe seu nome."; }
  if (!$err && !$locale0) { $err = "Selecione um idioma."; }

  // garante principal dentro da lista
  if (!in_array($locale0, $langs, true)) $langs[] = $locale0;
  // normaliza + filtra suportados
  $langs = array_values(array_intersect(array_map('normalize_locale', $langs), $all_locales));
  if (empty($langs)) $langs = [$locale0];

  if (!$err) {
    try {
      $pdo->beginTransaction();

      // === cria ou reaproveita user pelo e-mail ===
      // Se não veio e-mail, cria novo usuário sem e-mail (aceito) — evita unicidade.
      $user_id = null;
      if ($email !== '') {
        // tenta inserir; se já existir, reaproveita
        try {
          $uuid = generate_uuid();
          $insU = $pdo->prepare("INSERT INTO users (uuid,email,name,default_locale) VALUES (?,?,?,?)");
          $insU->execute([$uuid, $email, $name, $locale0]);
          $user_id = (int)$pdo->lastInsertId();
        } catch (PDOException $e) {
          if ($e->getCode() === '23000') {
            // e-mail já existe: reaproveita
            $q = $pdo->prepare("SELECT id, default_locale FROM users WHERE email=? LIMIT 1");
            $q->execute([$email]);
            $exist = $q->fetch();
            if (!$exist) { throw $e; }
            $user_id = (int)$exist['id'];
            // atualiza nome e default_locale se veio algo novo
            $pdo->prepare("UPDATE users SET name=COALESCE(NULLIF(?,''),name), default_locale=? WHERE id=?")
                ->execute([$name, $locale0, $user_id]);
          } else {
            throw $e;
          }
        }
      } else {
        // sem e-mail: cria novo usuário anon (email NULL)
        $uuid = generate_uuid();
        $insU = $pdo->prepare("INSERT INTO users (uuid,email,name,default_locale) VALUES (?,?,?,?)");
        $insU->execute([$uuid, null, $name, $locale0]);
        $user_id = (int)$pdo->lastInsertId();
      }

      // garante linha base em user_profiles
      $chk = $pdo->prepare("SELECT 1 FROM user_profiles WHERE user_id=? LIMIT 1");
      $chk->execute([$user_id]);
      if (!$chk->fetch()) {
        $pdo->prepare("INSERT INTO user_profiles (user_id, theme, content) VALUES (?,?,CAST(? AS JSON))")
            ->execute([$user_id, 'tap', json_encode(new stdClass())]);
      }

      // cria/atualiza conteúdo por idioma (idempotente)
      foreach ($langs as $lc) {
        $content = [
          'profile'=>[
            'name'     => $name,
            'title'    => '',
            'location' => '',
            'email'    => $email,
            'phone'    => $phone,
            'whatsapp' => $phone,
            'avatar'   => '',
            // acento por idioma (apenas estética inicial)
            'accent'   => ($lc==='en-US' ? '#2563eb' : ($lc==='es-419' ? '#f59e0b' : '#22c55e')),
          ],
          'links'        => new stdClass(),
          'posts'        => [],
          'photos'       => [],
          'videos'       => [],
          'reviews'      => ['google'=>'','facebook'=>'','trustpilot'=>''],
          'integrations' => ['lead_webhook'=>''],
        ];

        $pdo->prepare("INSERT INTO user_profiles_i18n (user_id,locale,content)
                       VALUES (?,?,CAST(? AS JSON))
                       ON DUPLICATE KEY UPDATE content=VALUES(content), updated_at=NOW()")
            ->execute([$user_id, $lc, json_encode($content, JSON_UNESCAPED_UNICODE)]);
      }

      // sempre gera um NOVO token público (quem perdeu o link ganha outro)
      $token = create_user_token($user_id, 'Public', 'public', null);

      // marca o convite como usado
      $pdo->prepare("UPDATE invites SET status='used', used_at=NOW(), used_by=? WHERE id=?")
          ->execute([$user_id, $invite['id']]);

      $pdo->commit();

      // autentica sessão do dono e redireciona para o editor
      $_SESSION['user_id']     = $user_id;
      $_SESSION['auth_at']     = time();
      $_SESSION['auth_method'] = 'invite';

      $edit_url = rtrim($cfg['base_url'], '/')
                . '/edit.php?k=' . urlencode($token)
                . '&lang=' . urlencode($locale0)
                . '&first=1';

      header('Location: ' . $edit_url);
      exit;

    } catch (Throwable $e) {
      if ($pdo->inTransaction()) $pdo->rollBack();
      $err = "Erro ao criar seu cartão: " . $e->getMessage();
    }

    // reforça repopulação se houve erro
    $prefill_name  = $name;
    $prefill_email = $email;
    $prefill_phone = $phone;
    $default_locale = $locale0;
  }
}

// ---- GET: formulário clean ----
?>
<!doctype html>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>GLJET Card — Convite</title>
<style>
  :root{ --bd:#e5e7eb; --fg:#0b1220; --mut:#666; --accent:#22c55e; }
  body{ font-family: system-ui, -apple-system, Segoe UI, Roboto, Ubuntu, Cantarell, Noto Sans, sans-serif; color:var(--fg); line-height:1.45; margin:0; }
  .wrap{ max-width:640px; margin:36px auto; padding:0 16px; }
  h1{ margin:0 0 8px; }
  .card{ border:1px solid var(--bd); border-radius:14px; padding:16px; }
  label{ display:block; margin:8px 0 4px; font-weight:600; }
  input,select{ width:100%; padding:12px; border:1px solid var(--bd); border-radius:12px; font:inherit; }
  fieldset{ border:1px solid var(--bd); border-radius:12px; padding:12px; }
  legend{ padding:0 6px; color:#111; }
  .row{ display:grid; grid-template-columns:1fr 1fr; gap:12px; }
  .mut{ color:#666; font-size:13px; }
  .err{ background:#fee2e2; color:#991b1b; border:1px solid #fecaca; padding:10px 12px; border-radius:10px; margin:8px 0; }
  .btn{ display:inline-block; padding:12px 16px; border-radius:12px; text-decoration:none; cursor:pointer; border:1px solid var(--bd); background:#fff; }
  .btn.primary{ background:var(--accent); color:#0b1220; border:0; font-weight:700; }
</style>

<div class="wrap">
  <h1>Bem-vindo(a) 👋</h1>
  <p class="mut">Preencha abaixo para criar seu cartão digital.</p>

  <?php if ($err): ?><div class="err">❌ <?php echo h($err); ?></div><?php endif; ?>

  <form method="post" action="?c=<?php echo urlencode($code); ?>" class="card" onsubmit="this.querySelector('button[type=submit]').disabled=true">
    <input type="hidden" name="__csrf" value="<?php echo h(csrf()); ?>">
    <input type="hidden" name="__create" value="1">

    <label>Nome</label>
    <input name="name" required value="<?php echo h($prefill_name); ?>">

    <div class="row">
      <div>
        <label>E-mail <span class="mut">(opcional)</span></label>
        <input name="email" type="email" value="<?php echo h($prefill_email); ?>">
      </div>
      <div>
        <label>Telefone/WhatsApp</label>
        <input name="phone" value="<?php echo h($prefill_phone); ?>">
      </div>
    </div>

    <fieldset>
      <legend>Idiomas</legend>
      <p class="mut" style="margin:6px 0">Idioma principal:</p>
      <select name="default_locale">
        <?php foreach ($all_locales as $lc): ?>
          <option value="<?php echo h($lc); ?>" <?php echo $lc===$default_locale?'selected':''; ?>>
            <?php echo h($lc); ?>
          </option>
        <?php endforeach; ?>
      </select>

      <p class="mut" style="margin:10px 0 6px">Quais idiomas você quer usar?</p>
      <?php foreach ($all_locales as $lc): ?>
        <label style="display:block;margin:4px 0;font-weight:500">
          <input type="checkbox" name="langs[]" value="<?php echo h($lc); ?>" <?php echo $lc===$default_locale?'checked':''; ?>>
          <?php echo h($lc); ?>
        </label>
      <?php endforeach; ?>
      <small class="mut">Você pode adicionar os outros idiomas depois.</small>
    </fieldset>

    <div style="margin-top:12px">
      <button class="btn primary" type="submit">Criar meu cartão</button>
    </div>
  </form>
</div>