const $ = s => document.querySelector(s);
let currentCmd = "/cpf";
let page = 1;
let lastArgs = "";

function setExplain(cmd){
  const map = {
    "/cpf": "Ex.: /cpf 033.303.479-10  → Busca todos os dados do CPF",
    "/numero": "Ex.: /numero +55 (21) 98765-4321 → Busca possíveis/antigos donos",
    "/nome1": "Ex.: /nome1 Joao Silva Santos → Nome exato",
    "/nome2": "Ex.: /nome2 Joao S Santos → Nome semelhante/abreviado",
    "/nome3": "Ex.: /nome3 Alexandre Nader 353737 → Nome + dígitos do CPF",
    "/nome4": "Ex.: /nome4 Alexandre Nader RJ → Nome + UF",
    "/bin": "Ex.: /bin 553636 → Info sobre BIN",
  };
  $("#explain").textContent = map[cmd] || "";
}

function refreshCmd(cmd){
  currentCmd = cmd;
  $("#commandPrefix").value = cmd;
  setExplain(cmd);
  $("#commandArgs").focus();
}

function fetchResults(){
  const args = $("#commandArgs").value.trim();
  lastArgs = args;
  $("#resultInfo").textContent = "Consultando...";
  $("#btnExport").disabled = true;
  fetch("/consulta/api/search.php", {
    method:"POST",
    headers:{"Content-Type":"application/json"},
    body: JSON.stringify({ command: currentCmd, args, page })
  })
  .then(r=>r.json())
  .then(j=>{
    $("#resultBox").textContent = j.text || "";
    $("#resultInfo").textContent = j.info || "";
    $("#pageLabel").textContent = `Página ${j.page || 1}`;
    $("#prevPage").disabled = !(j.has_prev);
    $("#nextPage").disabled = !(j.has_next);
    $("#btnExport").disabled = !(j.text && j.text.length);
  })
  .catch(e=>{
    $("#resultInfo").textContent = "Erro na consulta.";
  });
}

$("#btnSidebar").addEventListener("click", ()=>{
  if (window.matchMedia('(max-width: 992px)').matches) {
    $("#sidebar").classList.toggle("show");
  } else {
    $("#sidebar").classList.toggle("hide");
  }
});

document.querySelectorAll(".sidebar-link[data-cmd]").forEach(btn=>{
  btn.addEventListener("click", ()=>{
    const cmd = btn.getAttribute("data-cmd");
    refreshCmd(cmd);
    if (window.matchMedia('(max-width: 992px)').matches) {
      $("#sidebar").classList.remove("show");
    }
  });
});

$("#btnBuscar").addEventListener("click", ()=>{ page = 1; fetchResults(); });
$("#prevPage").addEventListener("click", ()=>{ if(page>1){ page--; fetchResults(); }});
$("#nextPage").addEventListener("click", ()=>{ page++; fetchResults(); });

document.addEventListener("DOMContentLoaded", ()=>{
  refreshCmd("/cpf");
});


(function () {
  const $btn    = document.getElementById('btnBuscar');
  const $prefix = document.getElementById('commandPrefix');  // ex.: "/cpf"
  const $args   = document.getElementById('commandArgs');    // ex.: "020.085.699-50"
  const $out    = document.getElementById('resultArea');     // <pre id="resultArea">...</pre>
  const $pager  = document.getElementById('pager');          // <div id="pager"></div>

  let isLoading = false;
  let inflight  = null; // AbortController

  function clearResults() {
    if ($out)   $out.textContent = '';
    if ($pager) $pager.innerHTML = '';
  }

  function setLoading(state) {
    isLoading = state;
    if (!$btn) return;
    if (state) {
      $btn.disabled = true;
      $btn.dataset.label = $btn.textContent;
      $btn.innerHTML = `
        <span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
        Consultando…
      `;
    } else {
      $btn.disabled = false;
      $btn.innerHTML = $btn.dataset.label || 'Procurar';
      delete $btn.dataset.label;
    }
  }

  async function doSearch(page = 1) {
    if (isLoading) return;

    const command = ($prefix?.value || '').trim() || '/cpf';
    const args    = ($args?.value   || '').trim();

    if (!args) {
      $args?.focus();
      return;
    }

    // 1) limpa já de cara
    clearResults();

    // 2) trava botão
    setLoading(true);

    try {
      inflight?.abort();
      inflight = new AbortController();

      const resp = await fetch('/consulta/api/search.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({ command, args, page }),
        signal: inflight.signal
      });

      const data = await resp.json().catch(() => ({}));
      const text = (data && typeof data.text === 'string') ? data.text : 'Erro na consulta';

      if ($out) $out.textContent = text;

      // (Opcional) paginação — se você já retorna has_prev/has_next:
      if ($pager) {
        $pager.innerHTML = '';
        if (data?.has_prev || data?.has_next) {
          const mk = (label, disabled, toPage) => {
            const b = document.createElement('button');
            b.className = 'btn btn-outline-secondary btn-sm';
            b.textContent = label;
            b.disabled = !!disabled || isLoading;
            b.addEventListener('click', () => doSearch(toPage));
            return b;
          };
          if (data?.has_prev) $pager.appendChild(mk('← Anterior', false, (data.page||1)-1));
          const p = document.createElement('span');
          p.className = 'small align-self-center mx-2';
          p.textContent = 'Página ${data.page||1}';
          $pager.appendChild(p);
          if (data?.has_next) $pager.appendChild(mk('Próxima →', false, (data.page||1)+1));
        }
      }
    } catch (e) {
      if ($out) $out.textContent = 'Erro na consulta';
    } finally {
      // 3) libera botão
      setLoading(false);
    }
  }

  // Clique do botão
  $btn?.addEventListener('click', (ev) => {
    ev.preventDefault();
    if (!isLoading) doSearch(1);
  });

  // Enter no campo de argumentos
  $args?.addEventListener('keydown', (ev) => {
    if (ev.key === 'Enter') {
      ev.preventDefault();
      if (!isLoading) doSearch(1);
    }
  });
})();