<?php
// /var/www/html/card/edit.php
session_start();
require __DIR__.'/db.php';
require __DIR__.'/token_helpers.php';
require __DIR__.'/locale_helpers.php';
require __DIR__.'/i18n.php';

$cfg = include __DIR__.'/config.php';
$pdo = db();

function h($s){ return htmlspecialchars($s ?? '', ENT_QUOTES, 'UTF-8'); }
function csrf_token(){
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
  return $_SESSION['csrf'];
}
function csrf_check($t){ return !empty($_SESSION['csrf']) && hash_equals($_SESSION['csrf'], $t); }

$token = $_GET['k'] ?? '';
if (!$token){ http_response_code(400); echo "Token ausente."; exit; }

$ctx = find_user_by_token($token);
if (!$ctx){ http_response_code(404); echo "Cartão não encontrado."; exit; }

$lang = normalize_locale($_GET['lang'] ?? ($cfg['default_locale'] ?? 'pt-BR'));

// Carrega conteúdo do idioma
$stmt = $pdo->prepare("SELECT content FROM user_profiles_i18n WHERE user_id=? AND locale=?");
$stmt->execute([$ctx['user_id'],$lang]);
$row = $stmt->fetch();
$data = $row ? json_decode($row['content'], true) : [];
if (!$data) {
  // base mínima
  $data = [
    'profile'=>['name'=>'','title'=>'','location'=>'','email'=>'','phone'=>'','whatsapp'=>'','avatar'=>'','accent'=>'#22c55e'],
    'links'=>[],
    'posts'=>[],
    'photos'=>[],
  ];
}

// Upload avatar (se enviado)
$upload_ok = null;
$err = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['__save'])) {
  if (!csrf_check($_POST['__csrf'] ?? '')) { http_response_code(403); exit('CSRF'); }

  // Atualiza campos texto
  $data['profile']['name']     = trim($_POST['name'] ?? '');
  $data['profile']['title']    = trim($_POST['title'] ?? '');
  $data['profile']['location'] = trim($_POST['location'] ?? '');
  $data['profile']['email']    = trim($_POST['email'] ?? '');
  $data['profile']['phone']    = trim($_POST['phone'] ?? '');
  $data['profile']['whatsapp'] = trim($_POST['whatsapp'] ?? '');
  $accent = trim($_POST['accent'] ?? '#22c55e');
  if (!preg_match('~^#[0-9a-fA-F]{6}$~', $accent)) $accent = '#22c55e';
  $data['profile']['accent'] = $accent;

  // Links sociais
  $data['links'] = [
    'website'   => trim($_POST['website'] ?? ''),
    'instagram' => trim($_POST['instagram'] ?? ''),
    'facebook'  => trim($_POST['facebook'] ?? ''),
    'tiktok'    => trim($_POST['tiktok'] ?? ''),
    'youtube'   => trim($_POST['youtube'] ?? ''),
    'linkedin'  => trim($_POST['linkedin'] ?? ''),
    'x'         => trim($_POST['x'] ?? ''),
  ];

  // Avatar (opcional)
  if (!empty($_FILES['avatar']['name']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
    $tmp  = $_FILES['avatar']['tmp_name'];
    $name = $_FILES['avatar']['name'];
    $size = (int)$_FILES['avatar']['size'];
    $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    $ok_ext = ['jpg','jpeg','png','gif','webp'];

    if ($size > 5*1024*1024) {
      $err = "Imagem muito grande (máx 5MB).";
    } elseif (!in_array($ext, $ok_ext, true)) {
      $err = "Formato não permitido.";
    } else {
      $uid = (int)$ctx['user_id'];
      $dir = __DIR__ . "/uploads/u{$uid}";
      if (!is_dir($dir)) {
        @mkdir($dir, 0775, true);
        @chown($dir, 'www-data'); @chgrp($dir, 'www-data');
      }
      $new = $dir . "/avatar.{$ext}";
      if (!@move_uploaded_file($tmp, $new)) {
        $err = "Falha ao salvar imagem.";
      } else {
        @chmod($new, 0664);
        $rel = rtrim($cfg['base_url'],'/')."/uploads/u{$uid}/avatar.{$ext}";
        $data['profile']['avatar'] = $rel;
        $upload_ok = true;
      }
    }
  }

  if (!$err) {
    // Salva no banco
    $pdo->prepare("INSERT INTO user_profiles_i18n (user_id,locale,content)
                   VALUES (?,?,CAST(? AS JSON))
                   ON DUPLICATE KEY UPDATE content=VALUES(content), updated_at=NOW()")
        ->execute([$ctx['user_id'],$lang,json_encode($data, JSON_UNESCAPED_UNICODE)]);

    // Redireciona pra evitar reenvio de POST
    header("Location: ".$cfg['base_url']."/edit.php?k=".urlencode($token)."&lang=".urlencode($lang)."&ok=1");
    exit;
  }
}

$ok = isset($_GET['ok']);
?>
<!doctype html><html lang="pt-BR"><head>
<meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
<title>Editar cartão — <?php echo h($lang); ?></title>
<link rel="stylesheet" href="<?php echo h($cfg['base_url']); ?>/assets/css/tap.css">
<style>
.form { max-width: 760px; margin: 24px auto; padding: 16px; font-family: system-ui; }
.form h1{ margin:0 0 10px }
.form label{ display:block; margin:8px 0 4px; color:#111; font-weight:600 }
.form input,.form textarea, .form select{
  width:100%; padding:10px; border:1px solid #ddd; border-radius:10px; font:inherit;
}
.row{ display:grid; grid-template-columns:1fr 1fr; gap:12px }
.actions{ margin-top:16px; display:flex; gap:10px; flex-wrap:wrap }
.btn{ padding:10px 14px; border-radius:10px; border:1px solid #ddd; background:#fff; cursor:pointer }
.btn.primary{ background:#22c55e; color:#0b1220; border:0; font-weight:700 }
.alert{ padding:10px 12px; border-radius:10px; background:#ecfccb; color:#3f6212; margin-bottom:10px }
.err{ background:#fee2e2; color:#991b1b }
.preview{ display:flex; align-items:center; gap:10px }
.preview img{ width:72px; height:72px; border-radius:50% }
.badge{ display:inline-block; padding:4px 10px; background:#f1f5f9; border-radius:999px; font-size:12px; color:#0b1220 }
</style>
</head><body>
<div class="form">
  <h1>Editar cartão <span class="badge"><?php echo h($lang); ?></span></h1>
  <p>Link público: <a href="<?php echo h($cfg['base_url'].'/'.$token.'?lang='.$lang); ?>" target="_blank"><?php echo h($cfg['base_url'].'/'.$token.'?lang='.$lang); ?></a></p>
  <p>
    Idioma:
    <?php foreach (($cfg['locales'] ?? ['pt-BR','en-US','es-419']) as $lc): ?>
      <a class="badge" href="?k=<?php echo urlencode($token); ?>&lang=<?php echo urlencode($lc); ?>"><?php echo h($lc); ?></a>
    <?php endforeach; ?>
  </p>

  <?php if ($ok): ?><div class="alert">✔️ Salvo com sucesso.</div><?php endif; ?>
  <?php if ($err): ?><div class="alert err">❌ <?php echo h($err); ?></div><?php endif; ?>
  <?php if ($upload_ok): ?><div class="alert">🖼️ Avatar atualizado.</div><?php endif; ?>

  <form method="post" enctype="multipart/form-data">
    <input type="hidden" name="__csrf" value="<?php echo h(csrf_token()); ?>">
    <input type="hidden" name="__save" value="1">

    <div class="row">
      <div>
        <label>Nome</label>
        <input name="name" value="<?php echo h($data['profile']['name']); ?>">
      </div>
      <div>
        <label>Atividade / Título</label>
        <input name="title" value="<?php echo h($data['profile']['title']); ?>">
      </div>
    </div>

    <div class="row">
      <div>
        <label>Localização</label>
        <input name="location" value="<?php echo h($data['profile']['location']); ?>">
      </div>
      <div>
        <label>Cor do tema (hex)</label>
        <input name="accent" value="<?php echo h($data['profile']['accent']); ?>" placeholder="#22c55e">
      </div>
    </div>

    <div class="row">
      <div>
        <label>E-mail</label>
        <input name="email" value="<?php echo h($data['profile']['email']); ?>">
      </div>
      <div>
        <label>Telefone</label>
        <input name="phone" value="<?php echo h($data['profile']['phone']); ?>">
      </div>
    </div>

    <div class="row">
      <div>
        <label>WhatsApp</label>
        <input name="whatsapp" value="<?php echo h($data['profile']['whatsapp']); ?>">
      </div>
      <div>
        <label>Avatar (jpg/png/webp até 5MB)</label>
        <div class="preview">
          <?php if (!empty($data['profile']['avatar'])): ?>
            <img src="<?php echo h($data['profile']['avatar']); ?>" alt="avatar">
          <?php endif; ?>
          <input type="file" name="avatar" accept=".jpg,.jpeg,.png,.gif,.webp">
        </div>
      </div>
    </div>

    <hr>

    <div class="row">
      <div><label>Website</label><input name="website" value="<?php echo h($data['links']['website'] ?? ''); ?>"></div>
      <div><label>Instagram</label><input name="instagram" value="<?php echo h($data['links']['instagram'] ?? ''); ?>"></div>
    </div>
    <div class="row">
      <div><label>Facebook</label><input name="facebook" value="<?php echo h($data['links']['facebook'] ?? ''); ?>"></div>
      <div><label>LinkedIn</label><input name="linkedin" value="<?php echo h($data['links']['linkedin'] ?? ''); ?>"></div>
    </div>
    <div class="row">
      <div><label>YouTube</label><input name="youtube" value="<?php echo h($data['links']['youtube'] ?? ''); ?>"></div>
      <div><label>X (Twitter)</label><input name="x" value="<?php echo h($data['links']['x'] ?? ''); ?>"></div>
    </div>
    <div class="row">
      <div><label>TikTok</label><input name="tiktok" value="<?php echo h($data['links']['tiktok'] ?? ''); ?>"></div>
      <div></div>
    </div>

    <div class="actions">
      <button class="btn primary" type="submit">Salvar</button>
      <a class="btn" target="_blank" href="<?php echo h($cfg['base_url'].'/'.$token.'?lang='.$lang); ?>">Ver cartão</a>
    </div>
  </form>
</div>
</body></html>